/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "common.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/tabs.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/browser.h"
#include "viewlog.h"
#include "actions.h"
#include "event.h"

#define WINDOW_WIDTH        630
#define WINDOW_HEIGHT       473
#define MAX_DISP_ENTRIES    18

struct xui_viewlog_s {
  gui_new_window_t      nw;

  xitk_widget_t        *tabs;
  int                   tabs_height;

  int                   real_num_entries;
  char                **log;

  xitk_widget_t        *browser, *refresh, *copy, *close;

  int                 (*exit) (xui_viewlog_t *viewlog);
};

/*
 * Leaving setup panel, release memory.
 */
static int _viewlog_exit (xui_viewlog_t *vl) {
  gui_window_delete (&vl->nw);
  free (vl->log);
  vl->nw.gui->viewlog = NULL;
  free (vl);
  return 1;
}

static void viewlog_exit (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_viewlog_t *vl = data;

  (void)w;
  (void)state;
  (void)modifier;
  vl->exit (vl);
}

static void viewlog_copy (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_viewlog_t *vl = data;
  int i;

  (void)w;
  (void)state;
  (void)modifier;
  for (i = 1; i <= vl->real_num_entries; i++)
    vl->log[i][-1] = '\n';
  xitk_clipboard_set_text (vl->copy, vl->log[0], vl->log[vl->real_num_entries] - vl->log[0]);
  for (i = 1; i <= vl->real_num_entries; i++)
    vl->log[i][-1] = 0;
}

/*
 *
 */
static void viewlog_change_section (xitk_widget_t *wx, void *data, int section, unsigned int modifier) {
  xui_viewlog_t *vl = data;
  char buf[512];
  /* NOTE: this returns lines in reverse order (newest first). */
  const char * const *log = (const char * const *)xine_get_log (vl->nw.gui->xine, section);
  const char * const *d;
  const char *dummy[3];
  char **e, *q;
  uint32_t n, s;

  (void)wx;
  (void)modifier;
  if (!(log && log[0])) {
    dummy[0] = buf;
    dummy[1] = _("log info");
    dummy[2] = NULL;
    snprintf (buf, sizeof (buf),
      _("There is no log entry for logging section '%s'.\n"),
      xine_get_log_names (vl->nw.gui->xine)[section]);
    log = dummy;
  }

  /* Compute log entries */
  for (n = s = 0, d = log; *d; d++) {
    const char *p = *d;
    do {
      uint32_t l = xitk_find_0_or_byte (p, '\n');
      s += l, p += l;
      n++;
      if (!*p)
        break;
      p++;
    } while (*p);
  }
  s += ++n;

  /* TODO: better work around     v----v libxine API BUG: log may change after return from xine_get_log (). */
  e = malloc (n * sizeof (*e) + s + 4096);
  if (!e)
    return;
  free (vl->log);
  vl->log = e;
  q = (char *)(e + n);

  while (d > log) {
    const char *p = *--d;
    *e++ = q;
    while (1) {
      while (*p & 0xe0)
        *q++ = *p++;
      if (!*p) {
        *q++ = 0;
        break;
      }
      if (*p == '\n') {
        p++;
        *q++ = 0;
        if (!*p)
          break;
        *e++ = q;
      } else {
        p++;
        *q++ = ' ';
      }
    }
  }
  *e = q;
  vl->real_num_entries = e - vl->log;

  xitk_browser_update_list (vl->browser,
    (const char * const *)vl->log, NULL, vl->real_num_entries,
    (vl->real_num_entries < MAX_DISP_ENTRIES) ? 0 : vl->real_num_entries - MAX_DISP_ENTRIES);
}

/*
 * Refresh current displayed log.
 */
static void viewlog_refresh (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_viewlog_t *vl = data;
  int section =  xitk_tabs_get_current_selected (vl->tabs);
  (void)w;
  (void)state;
  (void)modifier;
  if (section >= 0) {
    viewlog_change_section (NULL, vl, section, modifier);
  }
}

static int viewlog_event (void *data, const xitk_be_event_t *e) {
  xui_viewlog_t *vl = data;

  switch (e->type) {
    case XITK_EV_KEY_DOWN:
    case XITK_EV_KEY_UP:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        if (e->qual & (MODIFIER_SHIFT | MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD4 | MODIFIER_MOD5))
          break;
        if (e->utf8[1] == XITK_KEY_F5)
          return xitk_widget_key_event (vl->refresh, e, 1);
        if (e->utf8[1] == XITK_KEY_ESCAPE)
          return xitk_widget_key_event (vl->close, e, 1);
      } else if (e->utf8[0] == ('c' & 0x1f)) {
        if (e->qual & (MODIFIER_SHIFT | MODIFIER_META | MODIFIER_MOD4 | MODIFIER_MOD5))
          break;
        return xitk_widget_key_event (vl->copy, e, 1);
      }
      break;
    case XITK_EV_DEL_WIN:
      return vl->exit (vl);
    default: ;
  }
  return gui_handle_be_event (vl->nw.gui, e);
}

/*
 * Create viewlog window
 */
void viewlog_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_viewlog_t *vl;

  if (!gui)
    return;

  vl = gui->viewlog;
  if (mode == XUI_W_OFF) {
    if (!vl)
      return;
    vl->exit (vl);
    return;
  } else if (mode == XUI_W_ON) {
    if (vl) {
      gui_raise_window (gui, vl->nw.xwin);
      return;
    }
  } else { /* toggle */
    if (vl) {
      vl->exit (vl);
      return;
    }
  }

  vl =(xui_viewlog_t *)calloc (1, sizeof (*vl));
  if (!vl)
    return;
  vl->log = NULL;
  vl->exit = _viewlog_exit;

  /* Create window */
  vl->nw.gui = gui;
  vl->nw.title = _("xine Log Viewer");
  vl->nw.id = "viewlog";
  if (xitk_init_NULL ()) {
    vl->nw.skin = NULL;
    vl->nw.wfskin = NULL;
    vl->nw.adjust = NULL;
  }
  vl->nw.wr.x = 80;
  vl->nw.wr.y = 80;
  vl->nw.wr.width = WINDOW_WIDTH;
  vl->nw.wr.height = WINDOW_HEIGHT;
  if (gui_window_new (&vl->nw) < 0) {
    free (vl->log);
    free (vl);
    return;
  }

  {
    const char * const *log_sections = xine_get_log_names (vl->nw.gui->xine);
    unsigned int        log_section_count = xine_get_log_section_count (vl->nw.gui->xine);
    /* create log sections */
    xitk_tabs_widget_t tab = {
      .nw = { .wl = vl->nw.wl, .userdata = vl, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .num_entries = log_section_count,
      .entries     = log_sections,
      .callback    = viewlog_change_section
    };

    vl->tabs = xitk_noskin_tabs_create (&tab, 15, 24, WINDOW_WIDTH - 30, tabsfontname);
    if (vl->tabs) {
      vl->tabs_height = xitk_get_widget_height (vl->tabs) - 1;
      xitk_image_draw_rectangular_box (vl->nw.bg, 15, 24 + vl->tabs_height,
        WINDOW_WIDTH - 30, MAX_DISP_ENTRIES * 20 + 10, XITK_DRAW_OUTTER);
      xitk_window_set_background_image (vl->nw.xwin, vl->nw.bg);
    }

    viewlog_change_section (NULL, vl, 0, 0);
  }

  {
    xitk_browser_widget_t br = {
      .nw = { .wl = vl->nw.wl, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .browser = {
        .max_displayed_entries = MAX_DISP_ENTRIES,
        .num_entries           = vl->real_num_entries,
        .entries               = (const char * const *)vl->log,
      }
    };
    vl->browser = xitk_noskin_browser_create (&br,
      15 + 5, (24 + vl->tabs_height) + 5, WINDOW_WIDTH - (30 + 10), 20, -16, br_fontname);
  }
  if (ALIGN_DEFAULT != ALIGN_LEFT)
    xitk_browser_set_alignment (vl->browser, ALIGN_LEFT);

  {
    uint32_t style = XITK_DRAW_SAT (vl->nw.gui->gfx_saturation);
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = vl->nw.wl,
        .tips = "F5",
        .userdata = vl,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .label = _("Refresh"),
      .button_type = CLICK_BUTTON,
      .align = ALIGN_CENTER,
      .style = XITK_DRAW_R | XITK_DRAW_B | style,
      .callback = viewlog_refresh
    };
    char buf[80];

    vl->refresh = xitk_noskin_labelbutton_create (&lb, 15, WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, tabsfontname);

    snprintf (buf, sizeof (buf), "%s-C", xitk_gettext ("Ctrl"));
    lb.nw.tips  = buf;
    lb.label    = xitk_gettext ("Copy");
    lb.callback = viewlog_copy;
    lb.style    = XITK_DRAW_R | XITK_DRAW_G | style;
    vl->copy = xitk_noskin_labelbutton_create (&lb,
      (WINDOW_WIDTH - (100 + 2 * 15)) / 2, WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    lb.nw.tips  = "Esc";
    lb.style    = XITK_DRAW_R | style;
    lb.label    = _("Close");
    lb.callback = viewlog_exit;
    vl->close = xitk_noskin_labelbutton_create (&lb, WINDOW_WIDTH - (100 + 15), WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, tabsfontname);
  }

  vl->nw.key = xitk_be_register_event_handler ("viewlog", vl->nw.xwin, viewlog_event, vl, NULL, NULL);

  gui_raise_window (vl->nw.gui, vl->nw.xwin);

  vl->nw.gui->viewlog = vl;
}
